<?php

use App\Models\Domain;
use App\Models\NameServer;
use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $domain;
    public $nameServers = [];
    public $showAddModal = false;
    public $showEditModal = false;
    public $showDeleteModal = false;
    public $nameServerId = null;
    public $host = '';

    public function mount($domain)
    {
        $this->domain = Domain::with('nameServers', 'user')->findOrFail($domain);
        $this->fetchNameServers();
    }

    public function fetchNameServers()
    {
        $this->nameServers = $this->domain->nameServers()->get();
    }

    public function openAddModal()
    {
        $this->reset(['host', 'nameServerId']);
        $this->showAddModal = true;
    }

    public function addNameServer()
    {
        $this->validate([
            'host' => 'required|string|max:255|unique:name_servers,host,NULL,id,domain_id,' . $this->domain->id,
        ]);

        if ($this->domain->nameServers()->count() >= 3) {
            $this->addError('host', __('Maximum 3 name servers allowed.'));
            return;
        }

        $this->domain->nameServers()->create([
            'host' => $this->host,
        ]);

        $this->showAddModal = false;
        $this->fetchNameServers();
        $this->reset('host');
    }

    public function openEditModal($id)
    {
        $ns = NameServer::findOrFail($id);
        $this->nameServerId = $ns->id;
        $this->host = $ns->host;
        $this->showEditModal = true;
    }

    public function updateNameServer()
    {
        $this->validate([
            'host' => 'required|string|max:255|unique:name_servers,host,' . $this->nameServerId . ',id,domain_id,' . $this->domain->id,
        ]);

        $ns = NameServer::findOrFail($this->nameServerId);
        $ns->update(['host' => $this->host]);

        $this->showEditModal = false;
        $this->fetchNameServers();
        $this->reset(['host', 'nameServerId']);
    }

    public function openDeleteModal($id)
    {
        $this->nameServerId = $id;
        $this->showDeleteModal = true;
    }

    public function deleteNameServer()
    {
        NameServer::findOrFail($this->nameServerId)->delete();
        $this->showDeleteModal = false;
        $this->fetchNameServers();
        $this->reset('nameServerId');
    }
}; ?>

<section class="w-full">

    @include('partials.settings-heading', [
    'heading' => __($domain->name),
    'subheading' => __('Ditambahkan pada:  :date', ['date' => $domain->created_at->translatedFormat('d - m - Y')]),
    ])

    <div class="flex justify-between items-center mb-4">
        <div class="font-semibold">{{ __('Name Servers') }}</div>
        <flux:button variant="primary" wire:click="openAddModal" :disabled="$domain->nameServers()->count() >= 3">
            {{ __('Tambah Name Server') }}
        </flux:button>
    </div>

    <ul class="space-y-4">
        @forelse ($nameServers as $ns)
        <li class="p-4 bg-white dark:bg-gray-900 rounded shadow flex justify-between items-center">
            <span>{{ $ns->host }}</span>
            <div class="flex gap-2">
                <flux:button size="sm" variant="filled" wire:click="openEditModal({{ $ns->id }})">
                    {{ __('Edit') }}
                </flux:button>
                <flux:button size="sm" variant="danger" wire:click="openDeleteModal({{ $ns->id }})">
                    {{ __('Delete') }}
                </flux:button>
            </div>
        </li>
        @empty
        <li class="text-gray-500">{{ __('Tidak ada name server.') }}</li>
        @endforelse
    </ul>

    {{-- Add Modal --}}
    <flux:modal wire:model="showAddModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Tambah Name Server') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Tambahkan nama server ke domain.') }}</flux:text>
            </div>
            <form wire:submit.prevent="addNameServer" class="space-y-4">
                <flux:input wire:model.defer="host" :label="__('Host')" placeholder="ns1.example.com" required
                    autofocus />
                @error('host')
                <flux:text class="text-red-600 text-sm">{{ $message }}</flux:text>
                @enderror
                <div class="flex justify-end gap-2 pt-2">
                    <flux:button type="button" wire:click="$set('showAddModal', false)">
                        {{ __('Batal') }}
                    </flux:button>
                    <flux:button type="submit" variant="primary">
                        {{ __('Simpan') }}
                    </flux:button>
                </div>
            </form>
        </div>
    </flux:modal>

    {{-- Edit Modal --}}
    <flux:modal wire:model="showEditModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Edit Name Server') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Update name server host.') }}</flux:text>
            </div>
            <form wire:submit.prevent="updateNameServer" class="space-y-4">
                <flux:input wire:model.defer="host" :label="__('Host')" placeholder="ns1.example.com" required
                    autofocus />
                @error('host')
                <flux:text class="text-red-600 text-sm">{{ $message }}</flux:text>
                @enderror
                <div class="flex justify-end gap-2 pt-2">
                    <flux:button type="button" wire:click="$set('showEditModal', false)">
                        {{ __('Batal') }}
                    </flux:button>
                    <flux:button type="submit" variant="primary">
                        {{ __('Update') }}
                    </flux:button>
                </div>
            </form>
        </div>
    </flux:modal>

    {{-- Delete Modal --}}
    <flux:modal wire:model="showDeleteModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Hapus Name Server') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Anda yakin ingin mengapus name server ini?') }}</flux:text>
            </div>
            <div class="flex justify-end gap-2 pt-2">
                <flux:button type="button" wire:click="$set('showDeleteModal', false)">
                    {{ __('Batal') }}
                </flux:button>
                <flux:button variant="danger" wire:click="deleteNameServer">
                    {{ __('Hapus') }}
                </flux:button>
            </div>
        </div>
    </flux:modal>
</section>