<?php

use App\Models\Domain;
use Livewire\Volt\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    use WithPagination;
    public $showCreateModal = false;
    public $showEditModal = false;
    public $showDeleteModal = false;
    public $domainId = null;
    public $name = '';
    #[\Livewire\Attributes\Url]
    public $search = '';
    public $perPage = 10;

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function getDomainsProperty()
    {
        return Domain::with('nameServers')
            ->orderBy('created_at', 'desc')
            ->when($this->search, fn($q) => $q->where('name', 'like', "%{$this->search}%"))
            ->paginate($this->perPage);
    }

    public function openCreateModal()
    {
        $this->reset(['name', 'domainId']);
        $this->showCreateModal = true;
    }

    public function createDomain()
    {
        $this->validate([
            'name' => 'required|unique:domains,name|max:255',
        ], [
            'name.required' => __('Nama domain wajib diisi.'),
            'name.unique' => __('Domain ini sudah ada.'),
            'name.max' => __('Nama domain tidak boleh lebih dari 255 karakter.'),
        ]);

        Domain::create([
            'user_id' => Auth::id(),
            'name' => $this->name,
        ]);

        $this->showCreateModal = false;
        $this->reset('name');
    }

    public function openEditModal($id)
    {
        $domain = Domain::findOrFail($id);
        $this->domainId = $domain->id;
        $this->name = $domain->name;
        $this->showEditModal = true;
    }

    public function updateDomain()
    {
        $this->validate([
            'name' => 'required|unique:domains,name,' . $this->domainId . '|max:255',
        ], [
            'name.required' => __('Nama domain wajib diisi.'),
            'name.unique' => __('Domain ini sudah ada.'),
            'name.max' => __('Nama domain tidak boleh lebih dari 255 karakter.'),
        ]);

        $domain = Domain::findOrFail($this->domainId);
        $domain->update(['name' => $this->name]);

        $this->showEditModal = false;
        $this->reset(['name', 'domainId']);
    }

    public function openDeleteModal($id)
    {
        $this->domainId = $id;
        $this->showDeleteModal = true;
    }

    public function deleteDomain()
    {
        Domain::findOrFail($this->domainId)->delete();
        $this->showDeleteModal = false;
        $this->reset('domainId');
    }
}; ?>

<section class="w-full">
    @include('partials.settings-heading', [
        'heading' => __('Domain'),
        'subheading' => __('Daftar domain yang Anda kelola.'),
    ])
    <div class="flex justify-between items-center gap-4 mb-6">
        <div class="flex-1">
            <flux:input icon="magnifying-glass" wire:model.live.debounce.500ms="search" 
                placeholder="Cari berdasarkan nama domain..." class="w-1/2 md:w-lg lg:w-xl" />
        </div>
        <div class="shrink-0">
            <flux:button variant="primary" wire:click="openCreateModal">
                {{ __('Tambah Domain') }}
            </flux:button>
        </div>
    </div>


    <div class="space-y-4">
        @forelse ($this->domains as $domain)
        <div class="p-5 bg-white dark:bg-gray-800 rounded-lg shadow">
            <div class="flex justify-between items-start">
                <div>
                    <a href="{{route('domains.show', ['domain' => $domain->id])}}">
                        <h3 class="font-semibold text-lg text-gray-900 dark:text-white">{{ $domain->name }}</h3>
                        <div class="mt-2 text-sm text-gray-600 dark:text-gray-400">
                            {{ __('Name Servers : ') }}
                            <ul class="mt-1 space-y-1 pl-4">
                                @forelse ($domain->nameServers as $ns)
                                <li class="list-disc">{{ $ns->host }}</li>
                                @empty
                                <li class="text-gray-500 dark:text-gray-400 italic ">{{ __('Tidak ada name server') }}</li>
                                @endforelse
                            </ul>
                        </div>
                    </a>
                </div>
                <div class="flex gap-2">
                    <flux:button size="sm" variant="filled" wire:click="openEditModal({{ $domain->id }})">
                        {{ __('Edit') }}
                    </flux:button>
                    <flux:button size="sm" variant="danger" wire:click="openDeleteModal({{ $domain->id }})">
                        {{ __('Hapus') }}
                    </flux:button>
                </div>
            </div>
        </div>
        @empty
        <div class="p-5 bg-white dark:bg-gray-800 rounded-lg shadow text-center text-gray-500 dark:text-gray-400">
            {{ __('Domain tidak ditemukan.') }}
        </div>
        @endforelse
    </div>

    <div class="mt-6">
        {{ $this->domains->links() }}
    </div>

    {{-- Create Modal --}}
    <flux:modal wire:model="showCreateModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Tambah Domain') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Tambahkan domain baru ke sistem') }}</flux:text>
            </div>

            <form wire:submit.prevent="createDomain" class="space-y-4">
                <flux:input wire:model.defer="name" :label="__('Domain Name')" placeholder="example.com" required
                    autofocus />
                @error('name')
                <flux:text class="text-red-600 text-sm">{{ $message }}</flux:text>
                @enderror

                <div class="flex justify-end gap-2 pt-2">
                    <flux:spacer />
                    <flux:button type="button" wire:click="$set('showCreateModal', false)">
                        {{ __('Batal') }}
                    </flux:button>
                    <flux:button type="submit" variant="primary">
                        {{ __('Simpan') }}
                    </flux:button>
                </div>
            </form>
        </div>
    </flux:modal>

    {{-- Edit Modal --}}
    <flux:modal wire:model="showEditModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Edit Domain') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Update informasi domain') }}</flux:text>
            </div>

            <form wire:submit.prevent="updateDomain" class="space-y-4">
                <flux:input wire:model.defer="name" :label="__('Domain Name')" placeholder="example.com" required
                    autofocus />
                @error('name')
                <flux:text class="text-red-600 text-sm">{{ $message }}</flux:text>
                @enderror

                <div class="flex justify-end gap-2 pt-2">
                    <flux:button type="button" wire:click="$set('showEditModal', false)">
                        {{ __('Batal') }}
                    </flux:button>
                    <flux:button type="submit" variant="primary">
                        {{ __('Update') }}
                    </flux:button>
                </div>
            </form>
        </div>
    </flux:modal>

    {{-- Delete Modal --}}
    <flux:modal wire:model="showDeleteModal" class="md:w-96">
        <div class="space-y-6">
            <div>
                <flux:heading size="lg">{{ __('Hapus Domain') }}</flux:heading>
                <flux:text class="mt-2">{{ __('Anda yakin ingin menghapus domain ini?.') }}</flux:text>
            </div>

            <div class="flex justify-end gap-2 pt-2">
                <flux:button type="button" wire:click="$set('showDeleteModal', false)">
                    {{ __('Batal') }}
                </flux:button>
                <flux:button variant="danger" wire:click="deleteDomain">
                    {{ __('Hapus') }}
                </flux:button>
            </div>
        </div>
    </flux:modal>
</section>